import { NextRequest, NextResponse } from 'next/server';
import { openDB } from '@/src/lib/db';

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        const { id } = await params;
        const db = await openDB();

        const customer = await new Promise((resolve, reject) => {
            db.get('SELECT * FROM customers WHERE id = ?', [id], (err, row) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row);
                }
            });
        });

        if (!customer) {
            return NextResponse.json({ error: 'Customer not found' }, { status: 404 });
        }

        return NextResponse.json(customer);
    } catch (error) {
        console.error('Error fetching customer:', error);
        return NextResponse.json({ error: 'Failed to fetch customer' }, { status: 500 });
    }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        const { id } = await params;
        const body = await request.json();
        const now = new Date().toISOString();
        const db = await openDB();

        const result = await new Promise((resolve, reject) => {
            db.run(`
                UPDATE customers 
                SET name = ?, email = ?, phone = ?, company = ?, status = ?, stage = ?, 
                    assigned_sales_rep = ?, value = ?, notes = ?, updated_at = ?
                WHERE id = ?
            `, [
                body.name,
                body.email,
                body.phone,
                body.company,
                body.status,
                body.stage,
                body.assigned_sales_rep,
                body.value,
                body.notes,
                now,
                id
            ], function (err) {
                if (err) {
                    reject(err);
                } else {
                    resolve({ changes: this.changes });
                }
            });
        });

        if ((result as any).changes === 0) {
            return NextResponse.json({ error: 'Customer not found' }, { status: 404 });
        }

        const updatedCustomer = await new Promise((resolve, reject) => {
            db.get('SELECT * FROM customers WHERE id = ?', [id], (err, row) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row);
                }
            });
        });

        return NextResponse.json(updatedCustomer);
    } catch (error) {
        console.error('Error updating customer:', error);
        return NextResponse.json({ error: 'Failed to update customer' }, { status: 500 });
    }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        const { id } = await params;
        const db = await openDB();

        const result = await new Promise((resolve, reject) => {
            db.run('DELETE FROM customers WHERE id = ?', [id], function (err) {
                if (err) {
                    reject(err);
                } else {
                    resolve({ changes: this.changes });
                }
            });
        });

        if ((result as any).changes === 0) {
            return NextResponse.json({ error: 'Customer not found' }, { status: 404 });
        }

        return NextResponse.json({ message: 'Customer deleted successfully' });
    } catch (error) {
        console.error('Error deleting customer:', error);
        return NextResponse.json({ error: 'Failed to delete customer' }, { status: 500 });
    }
}